#ifndef CGDISPLAY_CONFIGURATION_H
#define CGDISPLAY_CONFIGURATION_H

#include <IOKit/IOKitLib.h>
#include <CoreGraphics/CGDirectDisplay.h>
#include <AvailabilityMacros.h>

/*Display Configuration Change Flags

Specify the configuration parameters passed to a display reconfiguration callback function.

enum {
   kCGDisplayBeginConfigurationFlag = (1 << 0),
   kCGDisplayMovedFlag              = (1 << 1),
   kCGDisplaySetMainFlag            = (1 << 2),
   kCGDisplaySetModeFlag            = (1 << 3),
   kCGDisplayAddFlag                = (1 << 4),
   kCGDisplayRemoveFlag             = (1 << 5),
   kCGDisplayEnabledFlag            = (1 << 8),
   kCGDisplayDisabledFlag           = (1 << 9),
   kCGDisplayMirrorFlag             = (1 << 10),
   kCGDisplayUnMirrorFlag           = (1 << 11)
};
typedef u_int32_t CGDisplayChangeSummaryFlags;
Constants
kCGDisplayBeginConfigurationFlag

The display configuration is about to change.

Available in Mac OS X v10.3 and later.
kCGDisplayMovedFlag

The location of the upper-left corner of the display in global display space has changed.

Available in Mac OS X v10.3 and later.
kCGDisplaySetMainFlag

The display is now the main display.

Available in Mac OS X v10.3 and later.
kCGDisplaySetModeFlag

The display mode has changed.

Available in Mac OS X v10.3 and later.
kCGDisplayAddFlag

The display has been added to the active display list.

Available in Mac OS X v10.3 and later.
kCGDisplayRemoveFlag

The display has been removed from the active display list.

Available in Mac OS X v10.3 and later.
kCGDisplayEnabledFlag

The display has been enabled.

Available in Mac OS X v10.3 and later.
kCGDisplayDisabledFlag

The display has been disabled.

Available in Mac OS X v10.3 and later.
kCGDisplayMirrorFlag

The display is now mirroring another display.

Available in Mac OS X v10.3 and later.
kCGDisplayUnMirrorFlag

The display is no longer mirroring another display.

Available in Mac OS X v10.3 and later.
Discussion

For information about how these constants are used, see the callback CGDisplayReconfigurationCallBack.*/
enum {
   kCGDisplayBeginConfigurationFlag = (1 << 0),
   kCGDisplayMovedFlag              = (1 << 1),
   kCGDisplaySetMainFlag            = (1 << 2),
   kCGDisplaySetModeFlag            = (1 << 3),
   kCGDisplayAddFlag                = (1 << 4),
   kCGDisplayRemoveFlag             = (1 << 5),
   kCGDisplayEnabledFlag            = (1 << 8),
   kCGDisplayDisabledFlag           = (1 << 9),
   kCGDisplayMirrorFlag             = (1 << 10),
   kCGDisplayUnMirrorFlag           = (1 << 11)
};
typedef u_int32_t CGDisplayChangeSummaryFlags;

/*CGDisplayReconfigurationCallBack

A client-supplied callback function that’s invoked whenever the configuration of a local display is changed.

typedef void (*CGDisplayReconfigurationCallBack) (
   CGDirectDisplayID display,
   CGDisplayChangeSummaryFlags flags,
   void *userInfo
);

If you name your function MyDisplayReconfigurationCallBack, you would declare it like this:

void MyDisplayReconfigurationCallBack (
   CGDirectDisplayID display,
   CGDisplayChangeSummaryFlags flags,
   void *userInfo
);
Parameters
display

The display being reconfigured.
flags

Flags that indicate which display configuration parameters are changing.
userInfo

The userInfo argument passed to the function CGDisplayRegisterReconfigurationCallback when the callback function is registered. 
Discussion

To register a display reconfiguration callback function, you call the function CGDisplayRegisterReconfigurationCallback. Quartz invokes your callback function when: 
Your application calls a function to reconfigure a local display.
Your application is listening for events in the event-processing thread, and another application calls a function to reconfigure a local display.
The user changes the display hardware configuration—for example, by disconnecting a display or changing a system preferences setting.

Before display reconfiguration, Quartz invokes your callback function once for each online display to indicate a pending configuration change. The flags argument is always set to kCGDisplayBeginConfigurationFlag. Other than the display ID, this callback does not carry other per-display information, as details of how a reconfiguration affects a particular device rely on device-specific behaviors which may not be exposed by a device driver.

After display reconfiguration, Quartz invokes your callback function once for each added, removed, and online display. At this time, all display state reported by Core Graphics, QuickDraw, and the Carbon Display Manager will be up to date. This callback runs after the Carbon Display Manager notification callbacks. The flags argument indicates how the display configuration has changed. Note that in the case of removed displays, calls into Quartz with the removed display ID will fail.

The following code example illustrates how to test for specific conditions:void MyDisplayReconfigurationCallBack (
   CGDirectDisplayID display,
   CGDisplayChangeSummaryFlags flags,
   void *userInfo)
{
    if (flags & kCGDisplayAddFlag) {
        // display has been added
    }
    else if (flags & kCGDisplayRemoveFlag) {
        // display has been removed
    }
}


Your callback function should avoid attempting to change display configurations, and should not raise exceptions or perform a non-local return such as calling longjmp. When you are finished using a callback registration, you should call the function CGDisplayRemoveReconfigurationCallback to remove it.*/
typedef void (*CGDisplayReconfigurationCallBack) (
   CGDirectDisplayID display,
   CGDisplayChangeSummaryFlags flags,
   void *userInfo
);


//function

/*CGDisplayUsesOpenGLAcceleration

Returns a Boolean value indicating whether Quartz is using OpenGL-based window acceleration (Quartz Extreme) to render in a display.

boolean_t CGDisplayUsesOpenGLAcceleration (
   CGDirectDisplayID display
);
Parameters
display

The display to access.
Return Value

If true, Quartz Extreme is used to render in the specified display; otherwise, false.
Discussion

Quartz Extreme is an OpenGL-based, hardware-accelerated window compositor available in Mac OS X version 10.2 and later. Quartz Extreme requires a minimum hardware configuration to operate.

The information this function provides is typically used to adjust the demands of drawing operations to the capabilities of the display hardware. For example, an application running on an unaccelerated system could disable live window-resizing.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGDisplayConfiguration.h*/
boolean_t CGDisplayUsesOpenGLAcceleration (
   CGDirectDisplayID display
);

/*CGDisplayIOServicePort

Returns the I/O Kit service port of the specified display.

io_service_t CGDisplayIOServicePort (
   CGDirectDisplayID display
);
Parameters
display

The display to access.
Return Value

The I/O Kit service port for the specified display.
Discussion

An I/O Kit service port can be passed to I/O Kit to obtain additional information about the display. 

The port is owned by the graphics system, and should not be destroyed. 
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGDisplayConfiguration.h*/
io_service_t CGDisplayIOServicePort (
   CGDirectDisplayID display
);

/*CGDisplayRegisterReconfigurationCallback

Registers a callback function to be invoked whenever a local display is reconfigured.

void CGDisplayRegisterReconfigurationCallback (
   CGDisplayReconfigurationCallBack proc,
   void * userInfo
);
Parameters
proc

A pointer to the callback function to be registered.
userInfo

A pointer to user-defined data, or NULL. The userInfo argument is passed back to the callback function each time it’s invoked. 
Discussion

Whenever local displays are reconfigured, the callback function you register is invoked twice for each display that’s added, removed, or currently online—once before the reconfiguration, and once after the reconfiguration. For more information, see the callback type CGDisplayReconfigurationCallBack. 

A callback function may be registered multiple times with different user-defined data pointers, resulting in multiple registration entries. For each registration, when notification is no longer needed you should remove the registration by calling the function CGDisplayRemoveReconfigurationCallback. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
CGDisplayConfiguration.h*/
void CGDisplayRegisterReconfigurationCallback (
   CGDisplayReconfigurationCallBack proc,
   void * userInfo
);

/*CGDisplayRemoveReconfigurationCallback

Removes the registration of a callback function that’s invoked whenever a local display is reconfigured. 

void CGDisplayRemoveReconfigurationCallback (
   CGDisplayReconfigurationCallBack proc,
   void * userInfo
);
Parameters
proc

A pointer to the callback function associated with the registration to be removed.
userInfo

A pointer to user-defined data associated with the registration to be removed, or NULL. This is the same pointer that’s passed to the function CGDisplayRegisterReconfigurationCallback when registering the callback. 
Discussion

When you call this function, the two arguments must match the registered entry to be removed.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
CGDisplayConfiguration.h*/

void CGDisplayRemoveReconfigurationCallback (
   CGDisplayReconfigurationCallBack proc,
   void * userInfo
);

/*CGDisplayUnitNumber

Returns the logical unit number of a display.

uint32_t CGDisplayUnitNumber (
   CGDirectDisplayID display
);
Parameters
display

The display to access.
Return Value

A logical unit number for the specified display.
Discussion

The logical unit number represents a particular node in the I/O Kit device tree associated with the display’s frame buffer. For a particular hardware configuration, this value will not change when the attached monitor is changed. 

The unit number will change if the I/O Kit device tree changes, as when hardware is reconfigured, drivers are replaced, or significant changes occur to I/O Kit, so it should not be assumed to be invariant across login sessions.

For more information about I/O Kit, see the Apple publication “I/O Kit Fundamentals”.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGDisplayConfiguration.h*/
uint32_t CGDisplayUnitNumber (
   CGDirectDisplayID display
);


#endif