package com.babukuma.util.dbunit;

import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.Driver;
import java.util.Properties;

import org.dbunit.database.IDatabaseConnection;
import org.dbunit.dataset.Column;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.xml.XmlDataSet;

/**
 * DataCreator의 공통메소드 정의
 * @author BABUKUMA
 */
public abstract class AbstractDataCreator {
    /**
     * JDBC Class Location
     */
    private String classLocation;

    /**
     * JDBC Driver Name
     */
    private String jdbcDriver;

    /**
     * DB Connect URL
     */
    private String url;

    /**
     * DB Connect User ID
     */
    private String userID;

    /**
     * DB Connect User Password
     */
    private String password;

    /**
     * 사용할 DB schema
     */
    private String schema;

    /**
     * @param classLocation JDBC Class Location
     * @param jdbcDriver JDBC Driver Name
     * @param url DB Connect URL
     * @param userID DB Connect User ID
     * @param password DB Connect User Password
     * @param schema 사용할 DB schema
     */
    public AbstractDataCreator(String classLocation, String jdbcDriver,
        String url, String userID, String password, String schema) {
        this.classLocation = classLocation;
        this.jdbcDriver = jdbcDriver;
        this.url = url;
        this.userID = userID;
        this.password = password;
        this.schema = schema;
    }

    /**
     * Connection을 생성한다.
     * @return Connection
     */
    private Connection getConnection() {
        Connection jdbcConnection = null;

        try {
            File path = new File(getClassLocation());
            URL[] urls = new URL[] {path.toURL()};

            ClassLoader cl = new URLClassLoader(urls);

            Driver driver = (Driver) cl.loadClass(getJdbcDriver())
                .newInstance();
            Properties prop = new Properties();
            prop.put("user", getUserID());
            prop.put("password", getPassword());
            jdbcConnection = driver.connect(getUrl(), prop);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return jdbcConnection;
    }

    /**
     * 각 DataBase에 해당하는 Connection을 얻어온다.
     * @param connection Connection
     * @return IDatabaseConnection
     */
    protected abstract IDatabaseConnection getDatabaseConnection(
        Connection connection);

    /**
     * @param isSelect SELECT문으로 생성하는지, 모든 테이블을 생성하는지.
     * @param query SELECT문
     * @param resultName 결과 테이블의 이름
     * @param savePath 생성될 XMl 파일의 패스
     * @return 성공여부
     * @throws Exception 데이터 생성 실패시
     */
    public boolean makeXmlData(boolean isSelect, String query,
        String resultName, String savePath) throws Exception {
        if (isSelect) {
            return selectedTableCreator(query, resultName, savePath);
        } else {
            return allTableCreator(savePath);
        }
    }

    /**
     * 모든 테이블을 XML 파일로 저장한다.
     * @param savePath xml파일 저장위치
     * @return boolean 성공여부
     * @throws Exception 실패
     */
    private boolean allTableCreator(String savePath) throws Exception {
        IDataSet databaseDataSet = getDatabaseConnection(getConnection())
            .createDataSet();

        XmlDataSet.write(databaseDataSet, new FileOutputStream(savePath));

        return true;
    }

    /**
     * SELECT문을 통하여 데이터를 생성할 경우
     * @param query SELECT문
     * @param resultName 생성될 결과의 테이블명
     * @param savePath XML파일 생성위치
     * @return 생성성공여부
     * @throws Exception 실패시 발생
     */
    private boolean selectedTableCreator(String query, String resultName,
        String savePath) throws Exception {
        PrintWriter writer = null;

        try {
            ITable dataTable = getDatabaseConnection(getConnection())
                .createQueryTable(resultName, query);
            writer = new PrintWriter(new FileOutputStream(savePath));

            writer.println("<?xml version='1.0' encoding='UTF-8'?>");
            writer.println("<dataset>");
            writer.println("\t<table name=\"" + resultName + "\">");

            // Column
            ITableMetaData metaData = dataTable.getTableMetaData();
            Column[] column = metaData.getColumns();
            int columnSize = column.length;

            for (int i = 0; i < columnSize; i++) {
                writer.println("\t\t<column>" + column[i].getColumnName()
                    + "</column>");
            }

            for (int i = 0; i < dataTable.getRowCount(); i++) {
                writer.println("\t\t<row>");

                for (int n = 0; n < columnSize; n++) {
                    Object obj = dataTable.getValue(i, column[n]
                        .getColumnName());

                    if (obj != null) {
                        writer.println("\t\t\t<value>" + obj.toString()
                            + "</value>");
                    } else {
                        writer.println("\t\t\t<null/>");
                    }
                }

                writer.println("\t\t</row>");
            }

            writer.println("\t</table>");
            writer.print("</dataset>");
        } catch (Exception e) {
            throw e;
        } finally {
            if (writer != null) {
                writer.close();
            }
        }
        return true;
    }

    /**
     * @return Returns the classLocation.
     */
    protected final String getClassLocation() {
        return classLocation;
    }

    /**
     * @return Returns the jdbcDriver.
     */
    protected final String getJdbcDriver() {
        return jdbcDriver;
    }

    /**
     * @return Returns the url.
     */
    protected final String getUrl() {
        return url;
    }

    /**
     * @return Returns the userID.
     */
    protected final String getUserID() {
        return userID;
    }

    /**
     * @return Returns the password.
     */
    protected final String getPassword() {
        return password;
    }

    /**
     * @return Returns the schema.
     */
    protected final String getSchema() {
        return schema;
    }
}
